// Offscreen document for screen capture
// This runs in an offscreen document context where getUserMedia is available

console.log('🎬 [Offscreen] Document loaded and ready');

let activeStream = null;
let activeVideo = null;

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('🎬 [Offscreen] Received message:', request.type);
  
  if (request.type === 'request_display_media') {
    // Use getDisplayMedia - doesn't need stream ID
    requestDisplayMedia()
      .then(() => {
        console.log('✅ [Offscreen] Sending success response');
        sendResponse({ success: true });
      })
      .catch(error => {
        console.error('❌ [Offscreen] Error requesting display media:', error);
        sendResponse({ success: false, error: error.message });
      });
    
    return true; // Keep channel open for async response
  } else if (request.type === 'capture_frame') {
    captureFrame()
      .then(screenshot => {
        sendResponse({ screenshot: screenshot });
      })
      .catch(error => {
        console.error('Error capturing frame:', error);
        sendResponse({ screenshot: null, error: error.message });
      });
    
    return true; // Keep channel open for async response
  } else if (request.type === 'stop_screen_capture') {
    stopScreenCapture();
    sendResponse({ success: true });
  }
});

async function requestDisplayMedia() {
  try {
    console.log('🎬 [Offscreen] Requesting getDisplayMedia...');
    
    // Stop any existing stream
    stopScreenCapture();
    
    // Use getDisplayMedia - works in offscreen documents
    activeStream = await navigator.mediaDevices.getDisplayMedia({
      video: {
        displaySurface: 'monitor' // Prefer full screen
      },
      audio: false
    });

    console.log('✅ [Offscreen] getDisplayMedia successful, creating video element...');

    // Create a video element to play the stream
    activeVideo = document.createElement('video');
    activeVideo.srcObject = activeStream;
    activeVideo.autoplay = true;
    activeVideo.muted = true;

    // Wait for video to be ready
    await new Promise((resolve, reject) => {
      activeVideo.onloadedmetadata = () => {
        console.log('📺 [Offscreen] Video metadata loaded');
        activeVideo.play().then(resolve).catch(reject);
      };
      // Timeout after 5 seconds
      setTimeout(() => reject(new Error('Video load timeout')), 5000);
    });
    
    console.log('✅ [Offscreen] Screen capture stream started:', activeVideo.videoWidth, 'x', activeVideo.videoHeight);
  } catch (error) {
    console.error('❌ [Offscreen] Error requesting display media:', error);
    console.error('❌ [Offscreen] Error details:', error.name, error.message);
    throw error;
  }
}

async function startScreenCapture(streamId) {
  try {
    console.log('🎬 [Offscreen] Starting screen capture with stream ID:', streamId);
    
    // Stop any existing stream
    stopScreenCapture();
    
    console.log('🎬 [Offscreen] Requesting getUserMedia IMMEDIATELY...');
    
    // Get the screen stream using the provided streamId
    // MUST call getUserMedia immediately before stream ID expires
    activeStream = await navigator.mediaDevices.getUserMedia({
      audio: false,
      video: {
        chromeMediaSource: 'desktop',
        chromeMediaSourceId: streamId
      }
    });

    console.log('✅ [Offscreen] getUserMedia successful, creating video element...');

    // Create a video element to play the stream
    activeVideo = document.createElement('video');
    activeVideo.srcObject = activeStream;
    activeVideo.autoplay = true;
    activeVideo.muted = true;

    // Wait for video to be ready
    await new Promise((resolve, reject) => {
      activeVideo.onloadedmetadata = () => {
        console.log('📺 [Offscreen] Video metadata loaded');
        activeVideo.play().then(resolve).catch(reject);
      };
      // Timeout after 5 seconds
      setTimeout(() => reject(new Error('Video load timeout')), 5000);
    });
    
    console.log('✅ [Offscreen] Screen capture stream started:', activeVideo.videoWidth, 'x', activeVideo.videoHeight);
  } catch (error) {
    console.error('❌ [Offscreen] Error starting screen capture:', error);
    console.error('❌ [Offscreen] Error details:', error.name, error.message);
    throw error;
  }
}

async function captureFrame() {
  if (!activeVideo || !activeStream) {
    throw new Error('Screen capture not started');
  }
  
  try {
    // Create canvas and capture the current frame
    const canvas = document.createElement('canvas');
    canvas.width = activeVideo.videoWidth;
    canvas.height = activeVideo.videoHeight;
    
    const ctx = canvas.getContext('2d');
    ctx.drawImage(activeVideo, 0, 0, canvas.width, canvas.height);

    // Convert canvas to base64 PNG
    const screenshot = canvas.toDataURL('image/png', 0.8);
    
    return screenshot;
  } catch (error) {
    console.error('Error capturing frame:', error);
    throw error;
  }
}

function stopScreenCapture() {
  if (activeStream) {
    activeStream.getTracks().forEach(track => track.stop());
    activeStream = null;
  }
  if (activeVideo) {
    activeVideo.srcObject = null;
    activeVideo = null;
  }
  console.log('Screen capture stream stopped');
}
